package furny.ga;

import java.util.logging.Level;
import java.util.logging.Logger;

import math.geom2d.AffineTransform2D;
import math.geom2d.Point2D;
import math.geom2d.polygon.Polygon2D;
import math.geom2d.polygon.Rectangle2D;

import org.apache.commons.lang3.ObjectUtils;

import com.jme3.scene.Geometry;
import com.jme3.scene.Node;
import com.jme3.scene.shape.Box;

import furny.entities.Furniture;

/**
 * This is one gene segment for a furniture with a position and rotation in the
 * room.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class FurnEntry implements Comparable<FurnEntry> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(FurnEntry.class
      .getName());

  private RoomVector vector;
  private Furniture furniture;

  private Polygon2D bounds;

  /**
   * Create a new furniture entry.
   * 
   * @param vector
   *          The room vector that represents location and rotation.
   * @param furniture
   *          The furniture.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public FurnEntry(final RoomVector vector, final Furniture furniture) {
    super();
    this.vector = vector;
    setFurniture(furniture);
  }

  /**
   * Creates a copy of the furnitures model and translates/rotates it.
   * 
   * @return The translated and rotated model.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public Node createTranslatedModel() {

    final Node model = new Node("Furniture Node");
    final Node n = furniture.getModel().clone(true);

    model.attachChild(n);

    // assure that the model is rotated around the zero point
    model.setLocalTranslation(0f, 0f, 0f);

    // rotate the model
    model.getLocalRotation().fromAngles(0f, vector.getRotation(), 0f);

    // move the model to the vectors location
    model.setLocalTranslation(vector.getX(), 0f, vector.getY());

    return model;
  }

  /**
   * Getter for the outter bounds of the furniture.
   * 
   * @return The bounds.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public Polygon2D getBounds() {
    return bounds;
  }

  /**
   * Updates the bounds shape. Must be called when the room vector or furniture
   * have been exchanged.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private void updateShape() {
    final float width = furniture.getMetaData().getWidth();
    final float length = furniture.getMetaData().getLength();
    final Point2D p = new Point2D(vector.getX() - (width / 2d), vector.getY()
        - (length / 2d));

    bounds = new Rectangle2D(p, width, length);
    bounds = bounds.transform(AffineTransform2D.createRotation(vector.getX(),
        vector.getY(), vector.getRotation()));

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine("Bounding of FurnEntry: "
          + bounds.getBoundingBox().toString());
    }
  }

  /**
   * Getter for the room vector.
   * 
   * @return The room vector.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public RoomVector getVector() {
    return vector;
  }

  /**
   * Setter for the room vector.
   * 
   * @param vector
   *          The room vector.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setVector(final RoomVector vector) {
    this.vector = vector;
    updateShape();
  }

  /**
   * Getter for the furniture.
   * 
   * @return The furniture.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public Furniture getFurniture() {
    return furniture;
  }

  /**
   * Setter for the furniture.
   * 
   * @param furniture
   *          The furniture.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setFurniture(final Furniture furniture) {
    if (this.furniture != furniture) {
      this.furniture = furniture;

      updateShape();
    }
  }

  @Override
  protected FurnEntry clone() {
    return new FurnEntry(vector, furniture);
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "(" + vector.toString() + ","
        + furniture.toString() + ")";
  }

  @Override
  public int compareTo(final FurnEntry o) {
    return hashCode() - o.hashCode();
  }

  @Override
  public boolean equals(final Object obj) {
    return obj != null && obj.getClass() == getClass()
        && obj.hashCode() == hashCode()
        && ((FurnEntry) obj).furniture.equals(furniture)
        && ((FurnEntry) obj).vector.equals(vector);
  }

  @Override
  public int hashCode() {
    return ObjectUtils.hashCodeMulti(furniture, vector);
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          arguments
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final Node n = new Node();
    n.attachChild(new Geometry("", new Box(2f, 2f, 3f)));
    final Furniture f = new Furniture("", n);

    final FurnEntry fe = new FurnEntry(new RoomVector(100, 200, 0), f);

    final FurnEntry fe2 = new FurnEntry(new RoomVector(100, 200, 100), f);

    System.out.println(fe.getBounds().getBoundingBox().getAsAWTRectangle2D()
        .intersects(fe2.getBounds().getBoundingBox().getAsAWTRectangle2D()));
  }
}
